
    // ************************* Main.c **************************//
    // *************** See RFC NNNN for details. *****************//
    /* Copyright (c) 2016, 2024, 2025 IETF Trust and the persons identified as
     * authors of the code.  All rights reserved. See fnv-private.h for terms of use 
     * and redistribution. */

#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>

    /* To do a thorough test you need to run with FNV_64bitIntegers defined and with 
     * it undefined */
#include "FNVconfig.h"
#include "fnv-private.h"
#include "FNV32.h"
#include "FNV64.h"
#include "FNV128.h"
#include "FNV256.h"
#include "FNV512.h"
#include "FNV1024.h"

    /* global variables */
char           *funcName = "funcName not set?";
const char     *errteststring = "foo";
int             Terr = -1;      /* Total errors */
int             verbose = 0;    /* verbose flag */
enum { FNV32selected = 0, FNV64selected, FNV128selected,
    FNV256selected, FNV512selected, FNV1024selected,
    FNVnone = -1
} selected = FNVnone;
#define NTestBytes 3
const uint8_t   errtestbytes[NTestBytes] = { (uint8_t) 1,
    (uint8_t) 2, (uint8_t) 3
};

    // initial teststring is null so initial result is offset_basis
const char     *teststring[] = {
    "",
    "a",
    "foobar",
    "Hello!\x01\xFF\xED"
};
#define NTstrings (sizeof(teststring)/sizeof(char *))

    // due to FNV-1 versus FNV1a, xor in final backslash separately
const char      BasisString[] = "chongo <Landon Curt Noll> /\\../";
FNV32context    e32Context;
uint32_t        eUint32 = 42;
#ifdef FNV_64bitIntegers
uint64_t        eUint64 = 42;
#endif
FNV64context    e64Context;
FNV128context   e128Context;
FNV256context   e256Context;
FNV512context   e512Context;
FNV1024context  e1024Context;
uint8_t         hash[FNV1024size];      /* largest size needed */
uint8_t         FakeBasis[FNV1024size];
uint8_t         ZeroBasis[FNV1024size];
char            tempFileNameTemplate[] = "tmp.XXXXXXXXXX";
const char     *tempFileName = 0;

    // ****************************************************************
    // local prototypes in alphabetic order
    // ****************************************************************
void            CommonTest(void);
void            ErrTestReport(void);
int             find_selected(const char *optarg);
void            HexPrint(int count, const uint8_t * ptr);
void            TestAll(void);
void            Test32(void);
void            Test64(void);
void            Test128(void);
void            Test256(void);
void            Test512(void);
void            Test1024(void);
void            TestNValue(const char *subfunc, // test calculated value
                           const char *string, int N,   // size
                           const uint8_t * was, const uint8_t should[N]);
int             TestR(const char *, int expect, int actual);    // test return code
void            usage(const char *argv0);       // print help message
void            ValueTestReport(void);  // print test results

#ifndef FNV_64bitIntegers
#undef uint64
#define uint64_t no_64_bit_integers
#endif                          /* FNV_64bitIntegers */

    // array of function pointers, etc.
struct {                        // sometimes indexed into by the enum variable
                                // "selected"
    int             length;
    void            (*Testfunc)(void);
    int             (*Stringfunc)(const char *, uint8_t *);     // string
    int             (*Blockfunc)(const void *, long int, uint8_t *);    // block
    int             (*Filefunc)(const char *, uint8_t *);       // file
    int             (*StringBasisfunc)
                    (const char *, uint8_t *, const uint8_t *); // stringBasis
    int             (*BlockBasisfunc)
                    (const void *, long int, uint8_t *, const uint8_t *);       // blockBasis
    int             (*FileBasisfunc)
                    (const char *, uint8_t *, const uint8_t *); // fileBlock
} funcmap[] = {                 // valid sizes
    {32, Test32, FNV32string, FNV32block, FNV32file,
     FNV32stringBasis, FNV32blockBasis, FNV32fileBasis},
    {64, Test64, FNV64string, FNV64block, FNV64file,
     FNV64stringBasis, FNV64blockBasis, FNV64fileBasis},
    {128, Test128, FNV128string, FNV128block, FNV128file,
     FNV128stringBasis, FNV128blockBasis, FNV128fileBasis},
    {256, Test256, FNV256string, FNV256block, FNV256file,
     FNV256stringBasis, FNV256blockBasis, FNV256fileBasis},
    {512, Test512, FNV512string, FNV512block, FNV512file,
     FNV512stringBasis, FNV512blockBasis, FNV512fileBasis},
    {1024, Test1024, FNV1024string, FNV1024block, FNV1024file,
     FNV1024stringBasis, FNV1024blockBasis, FNV1024fileBasis},
    {0, Test32, FNV32string, FNV32block, FNV32file}     // fence post
};

    // ****************************************************************
    // main
    // ****************************************************************
int
main(int argc, const char **argv)
{
    int             option;     // command line option letter
    int             i;
    uint16_t        endianness = 5 * 256 + 11;

    mkstemp(tempFileNameTemplate);
    tempFileName = tempFileNameTemplate;

    if (((uint8_t *) & endianness)[0] != 11)
        printf("Coded for Little Endian but computer seems\n"
               " to be Big Endian! Multi-byte integer results\n"
               " may be incorrect!\n");
    for (i = 0; i < FNV1024size; ++i) { // initialize a couple of arrays
        ZeroBasis[i] = 0;
        FakeBasis[i] = (uint8_t) i;
    }
    if (argc == 1) {            // if no arguments
        TestAll();
        if (tempFileName)
            unlink(tempFileName);
        exit(0);
    }

    // process command line options
    // *****************************************************************
    while ((option = getopt(argc, (char *const *) argv, ":af:ht:u:v"))
           != -1) {
        if (verbose)
            printf("Got option %c\n", option);
        switch (option) {
        case 'a':              // run all tests
            TestAll();
            break;
        case 'f':              // followed by name of file to hash
            if (selected == FNVnone) {
                printf("No hash size selected.\n");
                break;
            }
            printf("FNV-%i Hash of contents of file '%s':\n",
                   funcmap[selected].length, optarg);
            if (funcmap[selected].Filefunc(optarg, hash))
                printf("Hashing file '%s' fails: %s.\n", optarg, strerror(errno));
            else
                HexPrint(funcmap[selected].length / 8, hash);
            printf("\n");
            break;
        case 'h':              // help
            usage(argv[0]);
            break;
        case 't':              // follow by size of FNV to test, 0->all
            selected = find_selected(optarg);
            if (selected == FNVnone)
                printf("Bad argument to option -t\n"
                       "Valid sizes are 32, 64, 128," " 256, 512, and 1024\n");
            else
                funcmap[selected].Testfunc();   // invoke test
            break;
        case 'u':              // follow by size of FNV to use
            selected = find_selected(optarg);
            if (selected == FNVnone)
                printf("Bad argument to option -u\n"
                       "Valid sizes are 32, 64, 128," "256, 512, and 1024\n");
            break;
        case 'v':              // toggle Verbose flag
            if ((verbose ^= 1)) {
                printf("Verbose on.\n");
#ifdef FNV_64bitIntegers
                printf("Has 64-bit Integers. ");
#else
                printf("Does not have 64-bit integers. ");
#endif
                // also tests the TestR function
                funcName = "Testing TestR";
                TestR("should fail", 1, 2);
                TestR("should not have failed", 3, 3);
            } else
                printf("Verbose off.\n");
            break;
        case '?':              // 
            printf("Unknown option %c\n", optopt);
            usage(argv[0]);
            return 1;
        }                       /* end switch */
    }                           /* end while */
    if ((option == -1) && verbose)
        printf("No more options.\n");

    // Through all the options, now, if a size is set, encrypt any
    // other tokens on the command line
    // ******************************************************
    for (i = optind; i < argc; ++i) {
        int             rc;     // return code

        if (selected == FNVnone) {
            printf("No hash size selected.\n");
            break;              // out of for
        }
        rc = funcmap[selected].Stringfunc(argv[i], hash);
        if (rc)
            printf("FNV-%i of '%s' returns error %i\n",
                   funcmap[selected].length, argv[i], rc);
        else {
            printf("FNV-%i of '%s' is ", funcmap[selected].length, argv[i]);
            HexPrint(funcmap[selected].length / 8, hash);
            printf("\n");
        }
    }
    if (tempFileName)
        unlink(tempFileName);
    return 0;
}                               /* end main */

    /* Write to a temp file
     * **************************************************************** */
const char     *
WriteTemp(const char *str, long int iLen)
{
    FILE           *fp = fopen(tempFileName, "w");
    if (!fp) {
        printf("Cannot open tempfile: %s: %s\n", tempFileName, strerror(errno));
        return 0;
    }
    long int        ret = fwrite(str, 1, iLen, fp);
    fclose(fp);
    if (ret != iLen) {
        printf("Cannot write tempfile: %s: %s\n", tempFileName, strerror(errno));
        return 0;
    }
    return tempFileName;
}

    // ****************************************************************
    // Test status return code
    // ****************************************************************
int
TestR(const char *name, int expect, int actual)
{
    if (expect != actual) {
        printf("%s %s returned %i instead of %i.\n", funcName, name, actual, expect);
        ++Terr;                 /* increment error count */
    }
    return actual;
}                               /* end TestR */

    // ****************************************************************
    // General byte vector return value test
    // ****************************************************************
void
TestNValue(const char *subfunc, const char *string,     // usually what was hashed
           int N, const uint8_t was[N], const uint8_t should[N])
{
    if (memcmp(was, should, N) != 0) {
        ++Terr;
        printf("%s %s of '%s'", funcName, subfunc, string);
        printf(" computed ");
        HexPrint(N, was);
        printf(", expected ");
        HexPrint(N, should);
        printf(".\n");
    } else if (verbose) {
        printf("%s %s of '%s' computed ", funcName, subfunc, string);
        HexPrint(N, was);
        printf(" as expected.\n");
    }
}                               /* end TestNValue */

    // ****************************************************************
    // Reports on status/value returns
    // ****************************************************************
void
ErrTestReport(void)
{
    if (Terr)
        printf("%s test of error checks failed %i times.\n", funcName, Terr);
    else if (verbose)
        printf("%s test of error checks passed.\n", funcName);
}                               /* end ErrTestReport */

void
ValueTestReport(void)
{
    if (Terr)
        printf("%s test of return values failed %i times.\n", funcName, Terr);
    else
        printf("%s test of return values passed.\n", funcName);
}                               /* end ValueTestReport */

    // ****************************************************************
    // Verify the size of hash as a command line option argument
    // and return the index in funcmap[], -1 if not found.
    // ****************************************************************
int
find_selected(const char *optarg)
{
    int             argval, count;

    count = sscanf(optarg, "%i", &argval);
    if (count > 0) {
        int             i;
        for (i = 0; funcmap[i].length; ++i) {
            if (funcmap[i].length == argval) {
                return i;
            }                   /* end if */
        }                       /* end for */
    }
    return FNVnone;
}                               /* end find_selected */

    // ****************************************************************
    // Print some bytes as hexadecimal
    // ****************************************************************
void
HexPrint(int count, const uint8_t *ptr)
{
    for (int i = 0; i < count; ++i)
        printf("%02X", ptr[i]);
}                               /* end HexPrint */

    // ****************************************************************
    // Test all sizes
    // ****************************************************************
void
TestAll(void)
{
    for (int i = 0; funcmap[i].length; ++i)
        funcmap[i].Testfunc();
}                               /* end TestAll */

    // ****************************************************************
    // Common error check tests
    // ****************************************************************
void
CommonTest(void)
{
    TestR("string1b", fnvNull, funcmap[selected].Stringfunc((char *) 0, hash));
    TestR("string2b", fnvNull,
          funcmap[selected].Stringfunc(errteststring, (uint8_t *) 0));
    TestR("strBasis1b", fnvNull,
          funcmap[selected].StringBasisfunc((char *) 0, hash, FakeBasis));
    TestR("strBasis2b", fnvNull,
          funcmap[selected].StringBasisfunc(errteststring,
                                            (uint8_t *) 0, FakeBasis));
    TestR("strBasis3b", fnvNull,
          funcmap[selected].StringBasisfunc(errteststring, hash, (uint8_t *) 0));
    TestR("blk1", fnvNull, funcmap[selected].Blockfunc((uint8_t *) 0, 1, hash));
    TestR("blk2", fnvBadParam, funcmap[selected].Blockfunc(errtestbytes, -1, hash));
    TestR("blk3", fnvNull,
          funcmap[selected].Blockfunc(errtestbytes, 1, (uint8_t *) 0));
    TestR("blk1b", fnvNull,
          funcmap[selected].BlockBasisfunc((uint8_t *) 0, 1, hash, FakeBasis));
    TestR("blk2b", fnvBadParam,
          funcmap[selected].BlockBasisfunc(errtestbytes, -1, hash, FakeBasis));
    TestR("blk3b", fnvNull,
          funcmap[selected].BlockBasisfunc(errtestbytes, 1,
                                           (uint8_t *) 0, FakeBasis));
    TestR("blk4b", fnvNull,
          funcmap[selected].BlockBasisfunc(errtestbytes, 1, hash, (uint8_t *) 0));
    TestR("file1", fnvNull, funcmap[selected].Filefunc((char *) 0, hash));
    TestR("file2", fnvNull, funcmap[selected].Filefunc("foo.txt", (uint8_t *) 0));
    TestR("file1b", fnvNull,
          funcmap[selected].FileBasisfunc((char *) 0, hash, FakeBasis));
    TestR("file2b", fnvNull,
          funcmap[selected].FileBasisfunc("foo.txt", (uint8_t *) 0, FakeBasis));
    TestR("file3b", fnvNull,
          funcmap[selected].FileBasisfunc("foo.txt", hash, (uint8_t *) 0));
}                               /* end CommonTest */

    // ****************************************************************
    // Print command line help
    // ****************************************************************
void
usage(const char *argv0)
{
    printf("%s [-a] [-t nnn] [-u nnn] [-v] [-f filename] [token ...]\n"
           "  -a = run all tests\n"
           "  -f filename = hash file contents\n"
           "  -h = help, print this message\n"
           "  -t nnn = Test hash size nnn\n"
           "  -u nnn = Use hash size nnn\n"
           "  -v = toggle Verbose flag\n" "  Each token is hashed.\n", argv0);
}                               /* end usage */

    // ****************************************************************
    // Test Macros
    // ****************************************************************

    // test for return values
    // ************************
#define TestInit(INIT,CTX,CTXT)                               \
    TestR ( "init1", fnvSuccess, INIT ( &CTX ) );                 \
    TestR ( "init2", fnvNull, INIT ( (CTXT *)0 ) );

#define TestInitBasis(INITB,CTX,CTXT)                         \
    TestR ( "initB1", fnvSuccess, INITB (&CTX, FakeBasis ));      \
    TestR ( "initB2", fnvNull, INITB ( (CTXT *)0, hash ) );       \
    TestR ( "initB3", fnvNull, INITB ( &CTX, (uint8_t *)0 ));

#define TestBlockin(BLKIN,CTX,CTXT)                           \
    TestR ( "blockin1", fnvNull,                                  \
        BLKIN ( (CTXT *)0, errtestbytes, NTestBytes ) );          \
    TestR ( "blockin2", fnvNull,                                  \
        BLKIN ( &CTX, (uint8_t *)0, NTestBytes ) );               \
    TestR ( "blockin3", fnvBadParam,                              \
        BLKIN ( &CTX, errtestbytes, -1 ) );                       \
    TestR ( "blockin4", fnvStateError,                            \
        BLKIN ( &CTX, errtestbytes, NTestBytes ) );

#define TestStringin(STRIN,CTX,CTXT)                          \
    TestR ( "stringin1", fnvNull,                                 \
        STRIN ( (CTXT *)0, errteststring ) );                     \
    TestR ( "stringin2", fnvNull, STRIN ( &CTX, (char *)0 ) );    \
    TestR ( "stringin3", fnvStateError,                           \
        STRIN ( &CTX, errteststring ) );

#define TestFilein(FLIN,CTX,CTXT)                             \
    TestR ( "file1", fnvNull, FLIN ( (CTXT *)0, errteststring ) );\
    TestR ( "file2", fnvNull, FLIN ( &CTX, (char *)0 ) );         \
    TestR ( "file3", fnvStateError,                               \
        FLIN ( &CTX, errteststring ) );

#define TestResult(RSLT,CTX,CTXT)                             \
    TestR ( "result1", fnvNull, RSLT ( (CTXT *)0, hash ) );       \
    TestR ( "result2", fnvNull, RSLT ( &CTX, (uint8_t *)0  ) );   \
    TestR ( "result3", fnvStateError,                             \
                FNV128result ( &e128Context, hash  ) );

    // test return values for INT versions including non-std basis
    // *************************************************************
#define TestINT(STRINT,STRINTB,BLKINT,BLKINTB,INITINTB,       \
                    INTV,INTVT,ctxT)                              \
    TestR ( "string1i", fnvNull, STRINT ( (char *)0, &INTV ) );   \
    TestR ( "string2i", fnvNull,                                  \
            STRINT ( errteststring, (INTVT *)0 ) );               \
    TestR ("string3i", fnvNull, STRINTB ((char *)0, &INTV, INTV));\
    TestR ( "string4i", fnvNull,                                  \
            STRINTB (errteststring, (INTVT *)0, INTV));           \
    TestR ("block1i", fnvNull, BLKINT ( (uint8_t *)0, 1, &INTV ));\
    TestR ( "block2i", fnvBadParam,                               \
            BLKINT ( errtestbytes, -1, &INTV ) );                 \
    TestR ( "block3i", fnvNull,                                   \
            BLKINT ( errtestbytes, 1, (INTVT *)0 ) );             \
    TestR ( "block4i", fnvNull,                                   \
            BLKINTB ( (uint8_t *)0, 1, &INTV, INTV ) );           \
    TestR ( "block5i", fnvBadParam,                               \
            BLKINTB ( errtestbytes, -1, &INTV, INTV ) );    \
    TestR ( "block6i", fnvNull,                                   \
            BLKINTB ( errtestbytes, 1, (INTVT *)0, INTV ) );      \
    TestR ("initBasis1i", fnvNull, INITINTB ( (ctxT *)0, INTV ));

#define TestINTrf(RSLTINT,FILEINT,FILEINTB,                   \
                      ctx,ctxT,INTV,INTVT)                        \
    TestR ( "result1i", fnvNull, RSLTINT ( (ctxT *)0, &INTV ) );  \
    TestR ( "result2i", fnvNull, RSLTINT ( &ctx, (INTVT *)0  ) ); \
    TestR ( "result3i", fnvStateError, RSLTINT ( &ctx, &INTV  ) );\
    TestR ( "file1i", fnvNull, FILEINT ( (char *)0, &INTV ));     \
    TestR ( "file2i", fnvNull, FILEINT ( "foo.txt", (INTVT *)0 ));\
    TestR ("file3i", fnvNull, FILEINTB ( (char *)0, &INTV, INTV));\
    TestR ( "file4i", fnvNull,                                    \
            FILEINTB ( "foo.txt", (INTVT *)0, INTV ));

    // test to calculate standard basis from basis zero FNV-1
    // depends on zero basis making the initial multiply a no-op
    // *****************************
#define BasisZero(STRING,SIZ,VALUE)                           \
    err = TestR ( "fnv0s", fnvSuccess,                            \
                  STRING ( BasisString, hash, ZeroBasis ) );      \
    if ( err == fnvSuccess ) {                                    \
        hash[SIZ-1] ^= '\\';                                      \
        TestNValue ("fnv0sv", BasisString, SIZ, hash, VALUE[0]);  \
    }
#define BasisINTZero(STRINT,SIZ,VALUE,INTV,INTVT)             \
    err = TestR ( "fnv0s", fnvSuccess,                            \
                  STRINT ( BasisString, &INTV, (INTVT) 0 ) );     \
    if ( err == fnvSuccess ) {                                    \
        INTV ^= '\\';                                             \
        TestNValue ("fnv0svi", BasisString, SIZ,                  \
                    (uint8_t *)&INTV, (uint8_t *)&VALUE[0]);      \
    }

    // test for return hash values
    // *****************************
#define TestSTRBLKHash(STR,BLK,SVAL,BVAL,SZ)                  \
    if ( TestR ( "stringa", fnvSuccess,                           \
                 STR ( teststring[i], hash ) ) )                  \
        printf ( "  Index = %i\n", i );                           \
    else                                                          \
        TestNValue ( "stringb", teststring[i], SZ,                \
                     hash, (uint8_t *)&SVAL[i] );                 \
    if ( TestR ( "blocka", fnvSuccess, BLK ( teststring[i],       \
                 (long int)(strlen(teststring[i])+1), hash ) ) )  \
        printf ( "  Index = %i\n", i );                           \
    else                                                          \
        TestNValue ( "blockb", teststring[i], SZ,                 \
                     hash, (uint8_t *)&BVAL[i] );

    // Test incremental functions
    // ****************************
#define IncrHash(INIT,CTX,BLK,RSLT,INITB,STR,SZ,SVAL)         \
    err = TestR ( "inita", fnvSuccess, INIT ( &CTX ) );           \
    if ( err ) break;                                             \
    iLen = strlen ( teststring[i] );                              \
    err = TestR ("blockina", fnvSuccess,                          \
                 BLK ( &CTX, (uint8_t *)teststring[i], iLen/2 )); \
    if ( err ) break;                                             \
    if ( i & 1 ) {                                                \
        err = TestR ("basisra", fnvSuccess, RSLT ( &CTX, hash )); \
        if ( err ) break;                                         \
        err = TestR ("basisia", fnvSuccess, INITB ( &CTX, hash ));\
        if ( err ) break;                                         \
    }                                                             \
    err = TestR ( "stringina", fnvSuccess, STR ( &CTX,            \
                  teststring[i] + iLen/2 ) );                     \
    if ( err ) break;                                             \
    err = TestR ( "resulta", fnvSuccess, RSLT ( &CTX, hash ) );   \
    if ( err ) break;                                             \
    TestNValue ( "incrementala", teststring[i], SZ,               \
                 hash, (uint8_t *)&SVAL[i] );

    // test file hash
    // *****************************
#define TestFILEHash(FILE,BVAL,SZ)                            \
    err = TestR ( "fileafh", fnvSuccess,                          \
                   FILE ( WriteTemp(teststring[i], iLen),         \
                          hash ) );                               \
    if ( err ) break;                                             \
    TestNValue ( "filebfh", teststring[i], SZ, hash,              \
                 (uint8_t *)&BVAL[i] );

    // ****************************************************************
    // FNV32 Test
    // ****************************************************************
void
Test32(void)
{
    long int        iLen;
    uint32_t        FNV32svalues[NTstrings] = {
        0x811c9dc5, 0xe40c292c, 0xbf9cf968, 0xfd9d3881
    };
    uint32_t        FNV32bvalues[NTstrings] = {
        0x050c5d1f, 0x2b24d044, 0x0c1c9eb8, 0xbf7ff313
    };
    int             i, err;
    uint8_t         FNV32basisT[FNV32size] = { 0xC5, 0x9D, 0x1C, 0x81 };

    funcName = "FNV-32";
    selected = FNV32selected;
    /* test error checks */
    Terr = 0;
    TestInit(FNV32init, e32Context, FNV32context)
        TestInitBasis(FNV32initBasis, e32Context, FNV32context)
        CommonTest();
    TestINT(FNV32INTstring, FNV32INTstringBasis, FNV32INTblock,
            FNV32INTblockBasis, FNV32INTinitBasis, eUint32, uint32_t, FNV32context)
        e32Context.Computed = FNVclobber + FNV32state;
    TestBlockin(FNV32blockin, e32Context, FNV32context)
        TestStringin(FNV32stringin, e32Context, FNV32context)
        TestFilein(FNV32filein, e32Context, FNV32context)
        TestResult(FNV32result, e32Context, FNV32context)
        TestINTrf(FNV32INTresult, FNV32INTfile, FNV32INTfileBasis,
                  e32Context, FNV32context, eUint32, uint32_t)
        ErrTestReport();
    Terr = 0;
    err = TestR("fnv0s", fnvSuccess, FNV32stringBasis(BasisString, hash, ZeroBasis));
    if (err == fnvSuccess) {
        hash[0] ^= '\\';
        TestNValue("fnv0sv32", BasisString, FNV32size,
                   hash, (uint8_t *) & FNV32svalues[0]);
    }
    BasisINTZero(FNV32INTstringBasis, FNV32size, FNV32svalues, eUint32, uint32_t)
        for (i = 0; i < NTstrings; ++i) {
        /* test actual results int */
        err = TestR("stringai", fnvSuccess, FNV32INTstring(teststring[i], &eUint32));
        if (err == fnvSuccess)
            TestNValue("stringbi", teststring[i], FNV32size,
                       (uint8_t *) & eUint32, (uint8_t *) & FNV32svalues[i]);
        err = TestR("blockai", fnvSuccess,
                    FNV32INTblock((uint8_t *) teststring[i],
                                  (unsigned long) (strlen(teststring[i]) + 1),
                                  &eUint32));
        if (err == fnvSuccess)
            TestNValue("blockbi", teststring[i], FNV32size,
                       (uint8_t *) & eUint32, (uint8_t *) & FNV32bvalues[i]);
        /* test actual results byte */
        TestSTRBLKHash(FNV32string, FNV32block, FNV32svalues,
                       FNV32bvalues, FNV32size)
            /* now try testing the incremental stuff */
            IncrHash(FNV32init, e32Context, FNV32blockin, FNV32result,
                     FNV32initBasis, FNV32stringin, FNV32size, FNV32svalues)
            /* now try testing the incremental stuff int */
            err = TestR("initai", fnvSuccess, FNV32init(&e32Context));
        if (err)
            break;
        iLen = strlen(teststring[i]);
        err = TestR("blockinai", fnvSuccess,
                    FNV32blockin(&e32Context, (uint8_t *) teststring[i], iLen / 2));
        if (err)
            break;
        err = TestR("stringinai", fnvSuccess,
                    FNV32stringin(&e32Context, teststring[i] + iLen / 2));
        if (err)
            break;
        err = TestR("resultai", fnvSuccess, FNV32INTresult(&e32Context, &eUint32));
        if (err)
            break;
        TestNValue("incrementalai", teststring[i], FNV32size,
                   (uint8_t *) & eUint32, (uint8_t *) & FNV32svalues[i]);
        /* now try testing the incremental stuff byte basis */
        err = TestR("initab", fnvSuccess,
                    FNV32initBasis(&e32Context, (uint8_t *) & FNV32basisT));
        if (err)
            break;
        iLen = strlen(teststring[i]);
        err = TestR("blockinab", fnvSuccess,
                    FNV32blockin(&e32Context, (uint8_t *) teststring[i], iLen / 2));
        if (err)
            break;
        err = TestR("stringinab", fnvSuccess,
                    FNV32stringin(&e32Context, teststring[i] + iLen / 2));
        if (err)
            break;
        err = TestR("resultab", fnvSuccess, FNV32result(&e32Context, hash));
        if (err)
            break;
        TestNValue("incrementala", teststring[i], FNV32size,
                   hash, (uint8_t *) & FNV32svalues[i]);
        /* now try testing file hash int */
        err = TestR("fileafi", fnvSuccess,
                    FNV32INTfile(WriteTemp(teststring[i], iLen), &eUint32)
            );
        if (err)
            break;
        TestNValue("filebfi", teststring[i], FNV32size,
                   (uint8_t *) & eUint32, (uint8_t *) & FNV32svalues[i]);

        /* now try testing file hash byte */
        TestFILEHash(FNV32file, FNV32svalues, FNV32size)
    }                           // end for i
    ValueTestReport();
}                               /* end Test32 */

#ifdef FNV_64bitIntegers
    // ****************************************************************
    // Code for testing FNV64 using 64-bit integers
    // ****************************************************************
void
Test64(void)
{                               /* with 64-bit integers */
    long int        iLen;
    uint64_t        FNV64basisT = FNV64basis;
    uint64_t        FNV64svalues[NTstrings] = {
        0xcbf29ce484222325, 0xaf63dc4c8601ec8c, 0x85944171f73967e8,
        0xbd51ea7094ee6fa1
    };
    uint64_t        FNV64bvalues[NTstrings] = {
        0xaf63bd4c8601b7df, 0x089be207b544f1e4, 0x34531ca7168b8f38,
        0xa0a0fe4d1127ae93
    };
    int             i, err;

    funcName = "FNV-64";
    selected = FNV64selected;
    /* test error checks */
    Terr = 0;
    TestInit(FNV64init, e64Context, FNV64context)
        TestInitBasis(FNV64initBasis, e64Context, FNV64context)
        CommonTest();
    TestINT(FNV64INTstring, FNV64INTstringBasis, FNV64INTblock,
            FNV64INTblockBasis, FNV64INTinitBasis, eUint64, uint64_t, FNV64context)
        e64Context.Computed = FNVclobber + FNV64state;
    TestBlockin(FNV64blockin, e64Context, FNV64context)
        TestStringin(FNV64stringin, e64Context, FNV64context)
        TestFilein(FNV64filein, e64Context, FNV64context)
        TestResult(FNV64result, e64Context, FNV64context)
        TestINTrf(FNV64INTresult, FNV64INTfile, FNV64INTfileBasis,
                  e64Context, FNV64context, eUint64, uint64_t)
        ErrTestReport();
    /* test actual results int */
    Terr = 0;
    err = TestR("fnv0s", fnvSuccess, FNV64stringBasis(BasisString, hash, ZeroBasis));
    if (err == fnvSuccess) {
        hash[0] ^= '\\';
        TestNValue("fnv0sv64", BasisString, FNV64size,
                   hash, (uint8_t *) & FNV64svalues[0]);
    }
    BasisINTZero(FNV64INTstringBasis, FNV64size, FNV64svalues, eUint64, uint64_t)
        for (i = 0; i < NTstrings; ++i) {
        /* test actual results int */
        err = TestR("stringai", fnvSuccess, FNV64INTstring(teststring[i], &eUint64));
        if (err == fnvSuccess)
            TestNValue("stringbi", teststring[i], FNV64size,
                       (uint8_t *) & eUint64, (uint8_t *) & FNV64svalues[i]);
        err = TestR("blockai", fnvSuccess,
                    FNV64INTblock((uint8_t *) teststring[i],
                                  (unsigned long) (strlen(teststring[i]) + 1),
                                  &eUint64));
        if (err == fnvSuccess)
            TestNValue("blockbi", teststring[i], FNV64size,
                       (uint8_t *) & eUint64, (uint8_t *) & FNV64bvalues[i]);
        /* test actual results byte */
        TestSTRBLKHash(FNV64string, FNV64block, FNV64svalues,
                       FNV64bvalues, FNV64size)
            /* now try testing the incremental stuff */
            IncrHash(FNV64init, e64Context, FNV64blockin, FNV64result,
                     FNV64initBasis, FNV64stringin, FNV64size, FNV64svalues)
            /* now try testing the incremental stuff int */
            err = TestR("initai", fnvSuccess, FNV64init(&e64Context));
        if (err)
            break;
        iLen = strlen(teststring[i]);
        err = TestR("blockinai", fnvSuccess,
                    FNV64blockin(&e64Context, (uint8_t *) teststring[i], iLen / 2));
        if (err)
            break;
        err = TestR("stringinai", fnvSuccess,
                    FNV64stringin(&e64Context, teststring[i] + iLen / 2));
        if (err)
            break;
        err = TestR("resultai", fnvSuccess, FNV64INTresult(&e64Context, &eUint64));
        if (err)
            break;
        TestNValue("incrementalai", teststring[i], FNV64size,
                   (uint8_t *) & eUint64, (uint8_t *) & FNV64svalues[i]);
        /* now try testing the incremental stuff byte basis */
        err = TestR("initab", fnvSuccess,
                    FNV64initBasis(&e64Context, (uint8_t *) & FNV64basisT));
        if (err)
            break;
        iLen = strlen(teststring[i]);
        err = TestR("blockinab", fnvSuccess,
                    FNV64blockin(&e64Context, (uint8_t *) teststring[i], iLen / 2));
        if (err)
            break;
        err = TestR("stringinab", fnvSuccess,
                    FNV64stringin(&e64Context, teststring[i] + iLen / 2));
        if (err)
            break;
        err = TestR("resultab", fnvSuccess, FNV64result(&e64Context, hash));
        if (err)
            break;
        TestNValue("incrementala", teststring[i], FNV64size,
                   hash, (uint8_t *) & FNV64svalues[i]);
        /* now try testing file int */
        err = TestR("fileafi", fnvSuccess,
                    FNV64INTfile(WriteTemp(teststring[i], iLen), &eUint64)
            );
        if (err)
            break;
        TestNValue("filebfi", teststring[i], FNV64size,
                   (uint8_t *) & eUint64, (uint8_t *) & FNV64svalues[i]);
        /* now try testing file hash */
        TestFILEHash(FNV64file, FNV64svalues, FNV64size)
    }
    ValueTestReport();
}                               /* end Test64 */

#else

    // ****************************************************************
    // Code for testing FNV64 without 64-bit integers
    // ****************************************************************
void
Test64(void)
{                               /* without 64-bit integers */
    int             i, err;
    long int        iLen;
    uint8_t         FNV64svalues[NTstrings][FNV64size] = {
        {0xcb, 0xf2, 0x9c, 0xe4, 0x84, 0x22, 0x23, 0x25},
        {0xaf, 0x63, 0xdc, 0x4c, 0x86, 0x01, 0xec, 0x8c},
        {0x85, 0x94, 0x41, 0x71, 0xf7, 0x39, 0x67, 0xe8},
        {0xbd, 0x51, 0xea, 0x70, 0x94, 0xee, 0x6f, 0xa1}
    };
    uint8_t         FNV64bvalues[NTstrings][FNV64size] = {
        {0xaf, 0x63, 0xbd, 0x4c, 0x86, 0x01, 0xb7, 0xdf},
        {0x08, 0x9b, 0xe2, 0x07, 0xb5, 0x44, 0xf1, 0xe4},
        {0x34, 0x53, 0x1c, 0xa7, 0x16, 0x8b, 0x8f, 0x38},
        {0xa0, 0xa0, 0xfe, 0x4d, 0x11, 0x27, 0xae, 0x93}
    };

    funcName = "FNV-64";
    selected = FNV64selected;
    /* test error checks */
    Terr = 0;
    TestR("init1", fnvSuccess, FNV64init(&e64Context));
    CommonTest();
    TestInit(FNV64init, e64Context, FNV64context)
        TestInitBasis(FNV64initBasis, e64Context, FNV64context)
        e64Context.Computed = FNVclobber + FNV64state;
    TestBlockin(FNV64blockin, e64Context, FNV64context)
        TestStringin(FNV64stringin, e64Context, FNV64context)
        TestFilein(FNV64filein, e64Context, FNV64context)
        TestResult(FNV64result, e64Context, FNV64context)
        ErrTestReport();
    /* test actual results */
    Terr = 0;
    BasisZero(FNV64stringBasis, FNV64size, FNV64svalues)
        for (i = 0; i < NTstrings; ++i) {
        TestSTRBLKHash(FNV64string, FNV64block,
                       FNV64svalues, FNV64bvalues, FNV64size)
            /* try testing the incremental stuff */
            IncrHash(FNV64init, e64Context, FNV64blockin, FNV64result,
                     FNV64initBasis, FNV64stringin, FNV64size, FNV64svalues)
            /* now try testing file hash */
            TestFILEHash(FNV64file, FNV64svalues, FNV64size)
    }
    ValueTestReport();
}                               /* end Test64 */
#endif                          /* FNV_64bitIntegers */

    // ****************************************************************
    // Code for testing FNV128
    // ****************************************************************
void
Test128(void)
{
    int             i, err;
    long int        iLen;
    uint8_t         FNV128svalues[NTstrings][FNV128size] = {
        {0x6c, 0x62, 0x27, 0x2e, 0x07, 0xbb, 0x01, 0x42,
         0x62, 0xb8, 0x21, 0x75, 0x62, 0x95, 0xc5, 0x8d},
        {0xd2, 0x28, 0xcb, 0x69, 0x6f, 0x1a, 0x8c, 0xaf,
         0x78, 0x91, 0x2b, 0x70, 0x4e, 0x4a, 0x89, 0x64},
        {0x34, 0x3e, 0x16, 0x62, 0x79, 0x3c, 0x64, 0xbf,
         0x6f, 0x0d, 0x35, 0x97, 0xba, 0x44, 0x6f, 0x18},
        {0x74, 0x20, 0x2c, 0x60, 0x0b, 0x05, 0x1c, 0x16,
         0x5b, 0x1a, 0xca, 0xfe, 0xd1, 0x0d, 0x14, 0x19}
    };
    uint8_t         FNV128bvalues[NTstrings][FNV128size] = {
        {0xd2, 0x28, 0xcb, 0x69, 0x10, 0x1a, 0x8c, 0xaf,
         0x78, 0x91, 0x2b, 0x70, 0x4e, 0x4a, 0x14, 0x7f},
        {0x08, 0x80, 0x95, 0x45, 0x19, 0xab, 0x1b, 0xe9,
         0x5a, 0xa0, 0x73, 0x30, 0x55, 0xb7, 0x0e, 0x0c},
        {0xe0, 0x1f, 0xcf, 0x9a, 0x45, 0x4f, 0xf7, 0x8d,
         0xa5, 0x40, 0xf1, 0xb2, 0x32, 0x34, 0xb2, 0x88},
        {0xe2, 0x67, 0xa7, 0x41, 0xa8, 0x49, 0x8f, 0x82,
         0x19, 0xf7, 0xc7, 0x8b, 0x3b, 0x17, 0xba, 0xc3}
    };

    funcName = "FNV-128";
    selected = FNV128selected;
    /* test error checks */
    Terr = 0;
    TestInit(FNV128init, e128Context, FNV128context)
        TestInitBasis(FNV128initBasis, e128Context, FNV128context)
        CommonTest();
    e128Context.Computed = FNVclobber + FNV128state;
    TestBlockin(FNV128blockin, e128Context, FNV128context)
        TestStringin(FNV128stringin, e128Context, FNV128context)
        TestFilein(FNV128filein, e128Context, FNV128context)
        TestResult(FNV128result, e128Context, FNV128context)
        ErrTestReport();
    /* test actual results */
    Terr = 0;
    BasisZero(FNV128stringBasis, FNV128size, FNV128svalues)
        for (i = 0; i < NTstrings; ++i) {
        TestSTRBLKHash(FNV128string, FNV128block,
                       FNV128svalues, FNV128bvalues, FNV128size)
            /* try testing the incremental stuff */
            IncrHash(FNV128init, e128Context, FNV128blockin, FNV128result,
                     FNV128initBasis, FNV128stringin, FNV128size, FNV128svalues)
            /* now try testing file hash */
            TestFILEHash(FNV128file, FNV128svalues, FNV128size)
    }
    ValueTestReport();
}                               /* end Test128 */

    // ****************************************************************
    // Code for testing FNV256
    // ****************************************************************
void
Test256(void)
{
    int             i, err;
    long int        iLen;
    uint8_t         FNV256svalues[NTstrings][FNV256size] = {
        {0xdd, 0x26, 0x8d, 0xbc, 0xaa, 0xc5, 0x50, 0x36,
         0x2d, 0x98, 0xc3, 0x84, 0xc4, 0xe5, 0x76, 0xcc,
         0xc8, 0xb1, 0x53, 0x68, 0x47, 0xb6, 0xbb, 0xb3,
         0x10, 0x23, 0xb4, 0xc8, 0xca, 0xee, 0x05, 0x35},
        {0x63, 0x32, 0x3f, 0xb0, 0xf3, 0x53, 0x03, 0xec,
         0x28, 0xdc, 0x75, 0x1d, 0x0a, 0x33, 0xbd, 0xfa,
         0x4d, 0xe6, 0xa9, 0x9b, 0x72, 0x66, 0x49, 0x4f,
         0x61, 0x83, 0xb2, 0x71, 0x68, 0x11, 0x63, 0x7c},
        {0xb0, 0x55, 0xea, 0x2f, 0x30, 0x6c, 0xad, 0xad,
         0x4f, 0x0f, 0x81, 0xc0, 0x2d, 0x38, 0x89, 0xdc,
         0x32, 0x45, 0x3d, 0xad, 0x5a, 0xe3, 0x5b, 0x75,
         0x3b, 0xa1, 0xa9, 0x10, 0x84, 0xaf, 0x34, 0x28},
        {0x0c, 0x5a, 0x44, 0x40, 0x2c, 0x65, 0x38, 0xcf,
         0x98, 0xef, 0x20, 0xc4, 0x03, 0xa8, 0x0f, 0x65,
         0x9b, 0x80, 0xc9, 0xa5, 0xb0, 0x1a, 0x6a, 0x87,
         0x34, 0x2e, 0x26, 0x72, 0x64, 0x45, 0x67, 0xb1}
    };
    uint8_t         FNV256bvalues[NTstrings][FNV256size] = {
        {0x63, 0x32, 0x3f, 0xb0, 0xf3, 0x53, 0x03, 0xec,
         0x28, 0xdc, 0x56, 0x1d, 0x0a, 0x33, 0xbd, 0xfa,
         0x4d, 0xe6, 0xa9, 0x9b, 0x72, 0x66, 0x49, 0x4f,
         0x61, 0x83, 0xb2, 0x71, 0x68, 0x11, 0x38, 0x7f},
        {0xf4, 0xf7, 0xa1, 0xc2, 0xef, 0xd0, 0xe1, 0xe4,
         0xbb, 0x19, 0xe3, 0x45, 0x25, 0xc0, 0x72, 0x1a,
         0x06, 0xdd, 0x32, 0x8f, 0xa3, 0xd7, 0xa9, 0x14,
         0x39, 0xa0, 0x73, 0x43, 0x50, 0x1c, 0xf4, 0xf4},
        {0x6a, 0x7f, 0x34, 0xab, 0xc8, 0x5d, 0xe7, 0xd9,
         0x51, 0xb5, 0x15, 0x7e, 0xb5, 0x67, 0x2c, 0x59,
         0xb6, 0x04, 0x87, 0x65, 0x09, 0x47, 0xd3, 0x91,
         0xb1, 0x2d, 0x71, 0xe7, 0xfe, 0xf5, 0x53, 0x78},
        {0x3b, 0x97, 0x2c, 0x31, 0xbe, 0x84, 0x3a, 0x45,
         0x59, 0x02, 0x20, 0xd1, 0x12, 0x0d, 0x59, 0xe6,
         0xa3, 0x97, 0xa0, 0xc3, 0x34, 0xa1, 0xb9, 0x7d,
         0x5b, 0xff, 0x50, 0xa1, 0x0c, 0x3e, 0xca, 0x73}
    };

    funcName = "FNV-256";
    selected = FNV256selected;
    /* test error checks */
    Terr = 0;
    TestInit(FNV256init, e256Context, FNV256context)
        TestInitBasis(FNV256initBasis, e256Context, FNV256context)
        CommonTest();
    e256Context.Computed = FNVclobber + FNV256state;
    TestBlockin(FNV256blockin, e256Context, FNV256context)
        TestStringin(FNV256stringin, e256Context, FNV256context)
        TestFilein(FNV256filein, e256Context, FNV256context)
        TestResult(FNV256result, e256Context, FNV256context)
        ErrTestReport();
    /* test actual results */
    Terr = 0;
    BasisZero(FNV256stringBasis, FNV256size, FNV256svalues)
        for (i = 0; i < NTstrings; ++i) {
        TestSTRBLKHash(FNV256string, FNV256block,
                       FNV256svalues, FNV256bvalues, FNV256size)
            /* try testing the incremental stuff */
            IncrHash(FNV256init, e256Context, FNV256blockin, FNV256result,
                     FNV256initBasis, FNV256stringin, FNV256size, FNV256svalues)
            /* now try testing file hash */
            TestFILEHash(FNV256file, FNV256svalues, FNV256size)
    }
    ValueTestReport();
}                               /* end Test256 */

    // ****************************************************************
    // Code for testing FNV512
    // ****************************************************************
void
Test512(void)
{
    int             i, err;
    long int        iLen;
    uint8_t         FNV512svalues[NTstrings][FNV512size] = {
        {0xb8, 0x6d, 0xb0, 0xb1, 0x17, 0x1f, 0x44, 0x16,
         0xdc, 0xa1, 0xe5, 0x0f, 0x30, 0x99, 0x90, 0xac,
         0xac, 0x87, 0xd0, 0x59, 0xc9, 0x00, 0x00, 0x00,
         0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0d, 0x21,
         0xe9, 0x48, 0xf6, 0x8a, 0x34, 0xc1, 0x92, 0xf6,
         0x2e, 0xa7, 0x9b, 0xc9, 0x42, 0xdb, 0xe7, 0xce,
         0x18, 0x20, 0x36, 0x41, 0x5f, 0x56, 0xe3, 0x4b,
         0xac, 0x98, 0x2a, 0xac, 0x4a, 0xfe, 0x9f, 0xd9},
        {0xe4, 0x3a, 0x99, 0x2d, 0xc8, 0xfc, 0x5a, 0xd7,
         0xde, 0x49, 0x3e, 0x3d, 0x69, 0x6d, 0x6f, 0x85,
         0xd6, 0x43, 0x26, 0xec, 0x07, 0x00, 0x00, 0x00,
         0x00, 0x00, 0x00, 0x00, 0x00, 0x11, 0x98, 0x6f,
         0x90, 0xc2, 0x53, 0x2c, 0xaf, 0x5b, 0xe7, 0xd8,
         0x82, 0x91, 0xba, 0xa8, 0x94, 0xa3, 0x95, 0x22,
         0x53, 0x28, 0xb1, 0x96, 0xbd, 0x6a, 0x8a, 0x64,
         0x3f, 0xe1, 0x2c, 0xd8, 0x7b, 0x27, 0xff, 0x88},
        {0xb0, 0xec, 0x73, 0x8d, 0x9c, 0x6f, 0xd9, 0x69,
         0xd0, 0x5f, 0x0b, 0x35, 0xf6, 0xc0, 0xed, 0x53,
         0xad, 0xca, 0xcc, 0xcd, 0x8e, 0x00, 0x00, 0x00,
         0x4b, 0xf9, 0x9f, 0x58, 0xee, 0x41, 0x96, 0xaf,
         0xb9, 0x70, 0x0e, 0x20, 0x11, 0x08, 0x30, 0xfe,
         0xa5, 0x39, 0x6b, 0x76, 0x28, 0x0e, 0x47, 0xfd,
         0x02, 0x2b, 0x6e, 0x81, 0x33, 0x1c, 0xa1, 0xa9,
         0xce, 0xd7, 0x29, 0xc3, 0x64, 0xbe, 0x77, 0x88},
        {0x4f, 0xdf, 0x00, 0xec, 0xb9, 0xbc, 0x04, 0xdd,
         0x19, 0x38, 0x61, 0x8f, 0xe5, 0xc4, 0xfb, 0xb8,
         0x80, 0xa8, 0x2b, 0x15, 0xf5, 0xb6, 0xbd, 0x72,
         0x1e, 0xc2, 0xea, 0xfe, 0x03, 0xc4, 0x62, 0x48,
         0xf7, 0xa6, 0xc2, 0x47, 0x89, 0x92, 0x80, 0xd6,
         0xd2, 0xf4, 0x2f, 0xf6, 0xb4, 0x7b, 0xf2, 0x20,
         0x79, 0xdf, 0xd4, 0xbf, 0xe8, 0x7b, 0xf0, 0xbb,
         0x4e, 0x71, 0xea, 0xcb, 0x1e, 0x28, 0x77, 0x35}
    };
    uint8_t         FNV512bvalues[NTstrings][FNV512size] = {
        {0xe4, 0x3a, 0x99, 0x2d, 0xc8, 0xfc, 0x5a, 0xd7,
         0xde, 0x49, 0x3e, 0x3d, 0x69, 0x6d, 0x6f, 0x85,
         0xd6, 0x43, 0x26, 0xec, 0x28, 0x00, 0x00, 0x00,
         0x00, 0x00, 0x00, 0x00, 0x00, 0x11, 0x98, 0x6f,
         0x90, 0xc2, 0x53, 0x2c, 0xaf, 0x5b, 0xe7, 0xd8,
         0x82, 0x91, 0xba, 0xa8, 0x94, 0xa3, 0x95, 0x22,
         0x53, 0x28, 0xb1, 0x96, 0xbd, 0x6a, 0x8a, 0x64,
         0x3f, 0xe1, 0x2c, 0xd8, 0x7b, 0x28, 0x2b, 0xbf},
        {0x73, 0x17, 0xdf, 0xed, 0x6c, 0x70, 0xdf, 0xec,
         0x6a, 0xdf, 0xce, 0xd2, 0xa5, 0xe0, 0x4d, 0x7e,
         0xec, 0x74, 0x4e, 0x3c, 0xe9, 0x00, 0x00, 0x00,
         0x00, 0x00, 0x00, 0x00, 0x17, 0x93, 0x3d, 0x7a,
         0xf4, 0x5d, 0x70, 0xde, 0xf4, 0x23, 0xa3, 0x16,
         0xf1, 0x41, 0x17, 0xdf, 0x27, 0x2c, 0xd0, 0xfd,
         0x6b, 0x85, 0xf0, 0xf7, 0xc9, 0xbf, 0x6c, 0x51,
         0x96, 0xb3, 0x16, 0x0d, 0x02, 0x97, 0x5f, 0x38},
        {0x82, 0xf6, 0xe1, 0x04, 0x96, 0xde, 0x78, 0x34,
         0xb0, 0x8b, 0x21, 0xef, 0x46, 0x4c, 0xd2, 0x47,
         0x9e, 0x1d, 0x25, 0xe0, 0xca, 0x00, 0x00, 0x65,
         0xcb, 0x74, 0x80, 0x27, 0x39, 0xe0, 0xe5, 0x71,
         0x75, 0x22, 0xec, 0xf6, 0xd1, 0xf9, 0xa5, 0x2f,
         0x5f, 0xee, 0xfb, 0x4f, 0xab, 0x22, 0x73, 0xfd,
         0xe8, 0x31, 0x0f, 0x1b, 0x7b, 0x5c, 0x9a, 0x84,
         0x22, 0x48, 0xf4, 0xcb, 0xfb, 0x32, 0x27, 0x38},
        {0xfa, 0x7e, 0xb9, 0x1e, 0xfb, 0x64, 0x64, 0x11,
         0x8a, 0x73, 0x33, 0xbd, 0x96, 0x3b, 0xb6, 0x1f,
         0x2c, 0x6f, 0xe2, 0xe3, 0x6c, 0xd7, 0xd3, 0xe7,
         0x37, 0x28, 0xda, 0x57, 0x0c, 0x1f, 0xaf, 0xc3,
         0xd0, 0x6e, 0x4d, 0xd9, 0x53, 0x4a, 0x9f, 0xd4,
         0xa5, 0x2c, 0x43, 0x8b, 0xd2, 0x11, 0x69, 0x83,
         0x4a, 0xe6, 0x0d, 0x20, 0x7e, 0x0f, 0x8a, 0xf6,
         0x1a, 0xa1, 0x96, 0x25, 0x68, 0x37, 0xb8, 0x03}
    };

    funcName = "FNV-512";
    selected = FNV512selected;
    /* test error checks */
    Terr = 0;
    TestInit(FNV512init, e512Context, FNV512context)
        TestInitBasis(FNV512initBasis, e512Context, FNV512context)
        CommonTest();
    e512Context.Computed = FNVclobber + FNV512state;
    TestBlockin(FNV512blockin, e512Context, FNV512context)
        TestStringin(FNV512stringin, e512Context, FNV512context)
        TestFilein(FNV512filein, e512Context, FNV512context)
        TestResult(FNV512result, e512Context, FNV512context)
        ErrTestReport();
    /* test actual results */
    Terr = 0;
    BasisZero(FNV512stringBasis, FNV512size, FNV512svalues)
        for (i = 0; i < NTstrings; ++i) {
        TestSTRBLKHash(FNV512string, FNV512block,
                       FNV512svalues, FNV512bvalues, FNV512size)
            /* try testing the incremental stuff */
            IncrHash(FNV512init, e512Context, FNV512blockin, FNV512result,
                     FNV512initBasis, FNV512stringin, FNV512size, FNV512svalues)
            /* now try testing file hash */
            TestFILEHash(FNV512file, FNV512svalues, FNV512size)
    }
    ValueTestReport();
}                               /* end Test512 */

    // ****************************************************************
    // Code for testing FNV1024
    // ****************************************************************
void
Test1024(void)
{
    uint8_t         FNV1024svalues[NTstrings][FNV1024size] = {
        {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
         0x00, 0x5f, 0x7a, 0x76, 0x75, 0x8e, 0xcc, 0x4d,
         0x32, 0xe5, 0x6d, 0x5a, 0x59, 0x10, 0x28, 0xb7,
         0x4b, 0x29, 0xfc, 0x42, 0x23, 0xfd, 0xad, 0xa1,
         0x6c, 0x3b, 0xf3, 0x4e, 0xda, 0x36, 0x74, 0xda,
         0x9a, 0x21, 0xd9, 0x00, 0x00, 0x00, 0x00, 0x00,
         0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
         0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
         0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
         0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
         0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
         0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0xc6, 0xd7,
         0xeb, 0x6e, 0x73, 0x80, 0x27, 0x34, 0x51, 0x0a,
         0x55, 0x5f, 0x25, 0x6c, 0xc0, 0x05, 0xae, 0x55,
         0x6b, 0xde, 0x8c, 0xc9, 0xc6, 0xa9, 0x3b, 0x21,
         0xaf, 0xf4, 0xb1, 0x6c, 0x71, 0xee, 0x90, 0xb3},
        {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
         0x98, 0xd7, 0xc1, 0x9f, 0xbc, 0xe6, 0x53, 0xdf,
         0x22, 0x1b, 0x9f, 0x71, 0x7d, 0x34, 0x90, 0xff,
         0x95, 0xca, 0x87, 0xfd, 0xae, 0xf3, 0x0d, 0x1b,
         0x82, 0x33, 0x72, 0xf8, 0x5b, 0x24, 0xa3, 0x72,
         0xf5, 0x0e, 0x57, 0x00, 0x00, 0x00, 0x00, 0x00,
         0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
         0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
         0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
         0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
         0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
         0x00, 0x00, 0x00, 0x00, 0x07, 0x68, 0x5c, 0xd8,
         0x1a, 0x49, 0x1d, 0xbc, 0xcc, 0x21, 0xad, 0x06,
         0x64, 0x8d, 0x09, 0xa5, 0xc8, 0xcf, 0x5a, 0x78,
         0x48, 0x20, 0x54, 0xe9, 0x14, 0x70, 0xb3, 0x3d,
         0xde, 0x77, 0x25, 0x2c, 0xae, 0xf6, 0x95, 0xaa},
        {0x00, 0x00, 0x06, 0x31, 0x17, 0x5f, 0xa7, 0xae,
         0x64, 0x3a, 0xd0, 0x87, 0x23, 0xd3, 0x12, 0xc9,
         0xfd, 0x02, 0x4a, 0xdb, 0x91, 0xf7, 0x7f, 0x6b,
         0x19, 0x58, 0x71, 0x97, 0xa2, 0x2b, 0xcd, 0xf2,
         0x37, 0x27, 0x16, 0x6c, 0x45, 0x72, 0xd0, 0xb9,
         0x85, 0xd5, 0xae, 0x00, 0x00, 0x00, 0x00, 0x00,
         0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
         0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
         0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
         0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
         0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x42,
         0x70, 0xd1, 0x1e, 0xf4, 0x18, 0xef, 0x08, 0xb8,
         0xa4, 0x9e, 0x1e, 0x82, 0x5e, 0x54, 0x7e, 0xb3,
         0x99, 0x37, 0xf8, 0x19, 0x22, 0x2f, 0x3b, 0x7f,
         0xc9, 0x2a, 0x0e, 0x47, 0x07, 0x90, 0x08, 0x88,
         0x84, 0x7a, 0x55, 0x4b, 0xac, 0xec, 0x98, 0xb0},
        {0xf6, 0xf7, 0x47, 0xaf, 0x25, 0xa9, 0xde, 0x26,
         0xe8, 0xa4, 0x93, 0x43, 0x1e, 0x31, 0xb4, 0xa1,
         0xed, 0x2a, 0x92, 0x30, 0x4a, 0xf6, 0xca, 0x97,
         0x6b, 0xc1, 0xd9, 0x6f, 0xfc, 0xad, 0x35, 0x24,
         0x4e, 0x8d, 0x38, 0x5d, 0x55, 0xf4, 0x2f, 0xdc,
         0xc8, 0xf2, 0x99, 0x00, 0x00, 0x00, 0x00, 0x00,
         0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
         0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
         0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
         0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
         0x00, 0x00, 0x00, 0x00, 0xf7, 0xca, 0x87, 0xce,
         0x43, 0x22, 0x7b, 0x98, 0xc1, 0x44, 0x60, 0x7e,
         0x67, 0xcc, 0x50, 0xaf, 0x99, 0xbc, 0xc5, 0xd1,
         0x51, 0x4b, 0xb0, 0xd9, 0x23, 0xee, 0xde, 0xdd,
         0x69, 0xe8, 0xe7, 0x47, 0x02, 0x05, 0x08, 0x3a,
         0x0c, 0x02, 0x27, 0xd0, 0xcc, 0x69, 0xde, 0x23}
    };
    uint8_t         FNV1024bvalues[NTstrings][FNV1024size] = {
        {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
         0x98, 0xd7, 0xc1, 0x9f, 0xbc, 0xe6, 0x53, 0xdf,
         0x22, 0x1b, 0x9f, 0x71, 0x7d, 0x34, 0x90, 0xff,
         0x95, 0xca, 0x87, 0xfd, 0xae, 0xf3, 0x0d, 0x1b,
         0x82, 0x33, 0x72, 0xf8, 0x5b, 0x24, 0xa3, 0x72,
         0xf5, 0x0e, 0x38, 0x00, 0x00, 0x00, 0x00, 0x00,
         0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
         0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
         0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
         0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
         0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
         0x00, 0x00, 0x00, 0x00, 0x07, 0x68, 0x5c, 0xd8,
         0x1a, 0x49, 0x1d, 0xbc, 0xcc, 0x21, 0xad, 0x06,
         0x64, 0x8d, 0x09, 0xa5, 0xc8, 0xcf, 0x5a, 0x78,
         0x48, 0x20, 0x54, 0xe9, 0x14, 0x70, 0xb3, 0x3d,
         0xde, 0x77, 0x25, 0x2c, 0xae, 0xf6, 0x65, 0x97},
        {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xf4,
         0x6e, 0xf4, 0x1c, 0xd2, 0x3a, 0x4d, 0xcd, 0xd4,
         0x06, 0x83, 0x49, 0x63, 0xb7, 0x8e, 0x82, 0x24,
         0x1a, 0x6f, 0x5c, 0xb0, 0x6f, 0x40, 0x3c, 0xbd,
         0x5a, 0x7c, 0x89, 0x03, 0xce, 0xf6, 0xa5, 0xf4,
         0xfd, 0xd2, 0x95, 0x00, 0x00, 0x00, 0x00, 0x00,
         0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
         0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
         0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
         0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
         0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
         0x00, 0x00, 0x00, 0x0b, 0x7c, 0xd7, 0xfb, 0x20,
         0xc3, 0x63, 0x1d, 0xc8, 0x90, 0x39, 0x52, 0xe9,
         0xee, 0xb7, 0xf6, 0x18, 0x69, 0x8f, 0x4c, 0x87,
         0xda, 0x23, 0xad, 0x74, 0xb2, 0xc5, 0xf6, 0xf1,
         0xfe, 0xc4, 0xa6, 0x4b, 0x54, 0x66, 0x18, 0xa2},
        {0x00, 0x09, 0xdc, 0x92, 0x10, 0x75, 0xfd, 0x8a,
         0x5e, 0x3e, 0x1a, 0x37, 0x2c, 0x72, 0xa5, 0x9b,
         0xb1, 0x0c, 0xca, 0x1a, 0x94, 0xc8, 0xb2, 0x38,
         0x7d, 0x63, 0xa7, 0xef, 0xa7, 0xfc, 0xa7, 0xa7,
         0x17, 0xa6, 0x4e, 0x6c, 0x2d, 0x62, 0xfb, 0x61,
         0x78, 0xf7, 0x86, 0x00, 0x00, 0x00, 0x00, 0x00,
         0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
         0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
         0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
         0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
         0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x67, 0x08,
         0xf4, 0x4d, 0x00, 0x8a, 0xaa, 0xb0, 0x86, 0x57,
         0x49, 0x35, 0x50, 0x2c, 0x49, 0x08, 0x7c, 0x84,
         0x9b, 0xcb, 0xbe, 0xfa, 0x03, 0x3f, 0x45, 0x2a,
         0xf6, 0x38, 0x24, 0x26, 0xba, 0x5d, 0x3b, 0xb5,
         0x71, 0xb6, 0x46, 0x5b, 0x2a, 0xe8, 0xc8, 0xf0},
        {0xc8, 0x01, 0xf8, 0xe0, 0x8a, 0xe9, 0x1b, 0x18,
         0x0b, 0x98, 0xdd, 0x7d, 0x9f, 0x65, 0xce, 0xb6,
         0x87, 0xca, 0x86, 0x35, 0x8c, 0x69, 0x05, 0xf6,
         0x0a, 0x7d, 0x10, 0x14, 0xc1, 0x82, 0xb0, 0x4f,
         0xd6, 0x08, 0xa2, 0xca, 0x4d, 0xd6, 0x0a, 0x30,
         0x0a, 0x15, 0x68, 0x00, 0x00, 0x00, 0x00, 0x00,
         0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
         0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
         0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
         0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
         0x00, 0x00, 0x01, 0x80, 0x45, 0x14, 0x9a, 0xde,
         0x1c, 0x79, 0xab, 0xe3, 0xb7, 0x09, 0xa4, 0x06,
         0xf7, 0xd9, 0x20, 0x51, 0x69, 0xbe, 0xc5, 0x9b,
         0x12, 0x61, 0x40, 0xbc, 0xb9, 0x6f, 0x9d, 0x5d,
         0x3e, 0x2e, 0xa9, 0x1e, 0x21, 0xcd, 0xc2, 0x04,
         0x9f, 0x57, 0xbe, 0xcd, 0x00, 0x2d, 0x7c, 0x47}
    };
    long int        iLen;
    int             i, err;

    funcName = "FNV-1024";
    selected = FNV1024selected;
    /* test error checks */
    Terr = 0;
    TestInit(FNV1024init, e1024Context, FNV1024context)
        TestInitBasis(FNV1024initBasis, e1024Context, FNV1024context)
        CommonTest();
    e1024Context.Computed = FNVclobber + FNV1024state;
    TestBlockin(FNV1024blockin, e1024Context, FNV1024context)
        TestStringin(FNV1024stringin, e1024Context, FNV1024context)
        TestFilein(FNV1024filein, e1024Context, FNV1024context)
        TestResult(FNV1024result, e1024Context, FNV1024context)
        ErrTestReport();
    /* test actual results */
    Terr = 0;
    BasisZero(FNV1024stringBasis, FNV1024size, FNV1024svalues)
        for (i = 0; i < NTstrings; ++i) {
        TestSTRBLKHash(FNV1024string, FNV1024block,
                       FNV1024svalues, FNV1024bvalues, FNV1024size)
            /* try testing the incremental stuff */
            IncrHash(FNV1024init, e1024Context, FNV1024blockin,
                     FNV1024result, FNV1024initBasis,
                     FNV1024stringin, FNV1024size, FNV1024svalues)
            /* now try testing file hash */
            TestFILEHash(FNV1024file, FNV1024svalues, FNV1024size)
    }
    ValueTestReport();
}                               /* end Test1024 */
